# Linkie.dev - Technical Blog & Product Showcase

A professional Laravel 11 + Blade + TailwindCSS website combining a technical blog and product showcase platform with a dark mode design.

## Features

- **Technical Blog**: Tutorials, reviews, and affiliate content with category filtering
- **Product Showcase**: Scripts, templates, and tools with buy options
- **Admin Panel**: Full CRUD operations for posts and products
- **Dark Mode Design**: Bold typography with accent colors (#00B4D8, #1E3A8A)
- **SEO Optimized**: Meta tags, Open Graph, and Twitter Cards
- **Mobile Responsive**: Fully responsive design for all devices
- **Future Ready**: Architecture prepared for API, Stripe, and ThemeForest integration

## Tech Stack

- Laravel 11
- Blade Templates
- TailwindCSS v4 (via Vite)
- MySQL Database
- Laravel Breeze (Authentication)

## Installation

### Prerequisites

- PHP 8.2 or higher
- Composer
- Node.js and npm
- MySQL
- XAMPP (or similar local server)

### Step 1: Clone and Install Dependencies

```bash
cd linkie-dev
composer install
npm install
```

### Step 2: Environment Configuration

Copy the `.env.example` file to `.env`:

```bash
cp .env.example .env
```

Update the `.env` file with your database credentials:

```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=linkie_dev
DB_USERNAME=root
DB_PASSWORD=
```

### Step 3: Generate Application Key

```bash
php artisan key:generate
```

### Step 4: Run Migrations

```bash
php artisan migrate
```

### Step 5: Create Storage Link

```bash
php artisan storage:link
```

This creates a symbolic link from `public/storage` to `storage/app/public` for image uploads.

### Step 6: Build Assets

```bash
npm run build
```

For development with hot reload:

```bash
npm run dev
```

### Step 7: Create Admin User

Register a new user through the registration page at `/register`, or use tinker:

```bash
php artisan tinker
```

Then:

```php
$user = \App\Models\User::create([
    'name' => 'Admin',
    'email' => 'admin@linkie.dev',
    'password' => bcrypt('password'),
]);
```

### Step 8: Start Development Server

```bash
php artisan serve
```

Visit `http://localhost:8000` in your browser.

## Project Structure

```
linkie-dev/
├── app/
│   ├── Http/Controllers/
│   │   ├── HomeController.php
│   │   ├── BlogController.php
│   │   ├── ProductController.php
│   │   ├── AboutController.php
│   │   ├── ContactController.php
│   │   └── Admin/
│   │       ├── AdminPostController.php
│   │       └── AdminProductController.php
│   └── Models/
│       ├── Post.php
│       ├── Product.php
│       └── Contact.php
├── database/migrations/
│   ├── create_posts_table.php
│   ├── create_products_table.php
│   └── create_contacts_table.php
├── resources/views/
│   ├── layouts/
│   │   ├── main.blade.php (Public layout)
│   │   └── admin.blade.php (Admin layout)
│   ├── components/
│   │   ├── navbar.blade.php
│   │   ├── footer.blade.php
│   │   ├── blog-card.blade.php
│   │   └── product-card.blade.php
│   ├── home.blade.php
│   ├── blog/
│   │   ├── index.blade.php
│   │   └── show.blade.php
│   ├── products/
│   │   ├── index.blade.php
│   │   └── show.blade.php
│   ├── about.blade.php
│   ├── contact.blade.php
│   └── admin/
│       ├── dashboard.blade.php
│       ├── posts/
│       └── products/
└── routes/
    └── web.php
```

## Admin Panel

### Accessing the Admin Panel

1. Log in at `/login` with your admin credentials
2. Navigate to `/admin/dashboard`

### Admin Features

- **Dashboard**: Overview with statistics (posts, products, contacts)
- **Posts Management**: Create, edit, delete, and manage blog posts
  - Featured image upload
  - Category assignment
  - SEO meta tags
  - Publish date scheduling
- **Products Management**: Create, edit, delete, and manage products
  - Featured image and multiple images
  - Price setting
  - ThemeForest URL integration
  - Status management (active/inactive)

### Admin Routes

- `/admin/dashboard` - Admin dashboard
- `/admin/posts` - Posts list
- `/admin/posts/create` - Create new post
- `/admin/posts/{id}/edit` - Edit post
- `/admin/products` - Products list
- `/admin/products/create` - Create new product
- `/admin/products/{id}/edit` - Edit product

## Design System

### Color Palette

The design uses a dark mode-first approach with the following color scheme:

- **Background Primary**: `#121212` - Main dark background for pages
- **Background Card**: `#1a1a1a` - Darker background for cards and components
- **Background Darker**: `#0a0a0a` - Even darker for contrast sections
- **Text Primary**: `#E0E0E0` - Primary text color (light gray)
- **Text Muted**: `#E0E0E0/70` - Muted text with 70% opacity
- **Text Dim**: `#E0E0E0/50` - Dimmed text with 50% opacity
- **Accent**: `#00B4D8` - Cyan accent color for CTAs, links, and hover states
- **Accent Dark**: `#1E3A8A` - Darker blue accent for gradients and contrast

### Typography

The website uses a bold typography system with Google Fonts:

- **Headings**: Montserrat/Inter/Cairo (Bold, 700-900 weight)
  - Used for all headings (h1, h2, h3)
  - Applied via `.font-heading` utility class
  - Creates strong visual hierarchy
  
- **Body Text**: Inter/Cairo (Regular, 400 weight)
  - Used for paragraphs and body content
  - Ensures excellent readability on dark backgrounds

### TailwindCSS Classes & Comments

All Blade templates include comprehensive inline comments explaining Tailwind classes:

**Comment Format:**
```blade
{{-- Section Description: Brief explanation of the section --}}
<div class="flex items-center justify-between">
    {{-- Element Description: Detailed explanation of classes used --}}
    <h1 class="text-4xl font-bold font-heading text-white">Title</h1>
</div>
```

**What's Documented:**
- **Layout Choices**: Flexbox vs Grid, spacing decisions, alignment
- **Color Usage**: Why specific colors are used (background, text, accent)
- **Responsive Breakpoints**: Mobile-first approach (sm, md, lg breakpoints)
- **Hover Effects**: Transition animations and accent color changes
- **Component Structure**: How reusable components are structured

**Example Comment:**
```blade
{{-- Card Container: Dark card background (#1a1a1a), rounded corners, overflow hidden, border with hover accent color, smooth transitions, shadow on hover --}}
<div class="bg-[#1a1a1a] rounded-lg overflow-hidden border border-gray-800 hover:border-[#00B4D8] transition-all duration-300 hover:shadow-lg hover:shadow-[#00B4D8]/20">
```

This documentation helps developers understand design decisions and maintain consistency.

## Database Schema

### Posts Table

- `id`, `title`, `slug`, `excerpt`, `content`, `featured_image`, `category`, `meta_title`, `meta_description`, `published_at`, `timestamps`

### Products Table

- `id`, `name`, `slug`, `description`, `short_description`, `price`, `featured_image`, `images` (JSON), `themeforest_url`, `status`, `timestamps`

### Contacts Table

- `id`, `name`, `email`, `subject`, `message`, `timestamps`

## SEO Features

- Meta title and description for each blog post
- Open Graph tags for social media sharing
- Twitter Card support
- SEO-friendly URLs with slugs
- Semantic HTML structure

## Future Expansion

The project architecture is designed to be easily extensible. Here are planned enhancements:

### API Integration

The project structure is ready for API routes. To implement:

1. **Create API Routes**: Add `routes/api.php` with API endpoints
2. **API Controllers**: Create controllers in `app/Http/Controllers/Api/`
3. **Authentication**: Use Laravel Sanctum for API token authentication
4. **API Resources**: Use Laravel API Resources for consistent JSON responses
5. **Rate Limiting**: Implement rate limiting for API endpoints

**Example Structure:**
```
routes/api.php
app/Http/Controllers/Api/
├── PostController.php
├── ProductController.php
└── AuthController.php
```

### Stripe Payment Integration

For direct product purchases (when `themeforest_url` is not set):

1. **Install Stripe SDK**: `composer require stripe/stripe-php`
2. **Environment Variables**: Add Stripe keys to `.env`:
   ```env
   STRIPE_KEY=pk_test_...
   STRIPE_SECRET=sk_test_...
   STRIPE_WEBHOOK_SECRET=whsec_...
   ```
3. **Payment Controller**: Create `app/Http/Controllers/PaymentController.php`
4. **Checkout Routes**: Add checkout and webhook routes
5. **Order Management**: Create orders table and model
6. **Email Notifications**: Send purchase confirmations

**Database Schema Addition:**
```php
Schema::create('orders', function (Blueprint $table) {
    $table->id();
    $table->foreignId('user_id')->nullable();
    $table->foreignId('product_id');
    $table->string('stripe_payment_intent_id');
    $table->decimal('amount', 10, 2);
    $table->string('status');
    $table->timestamps();
});
```

### ThemeForest Integration

Products already have `themeforest_url` field. To enhance:

1. **ThemeForest API**: Integrate ThemeForest API for product sync
2. **Affiliate Tracking**: Add affiliate ID tracking to URLs
3. **Sales Analytics**: Track clicks and conversions
4. **Auto-sync**: Periodic sync of product data from ThemeForest

**Enhancement Example:**
```php
// Add to Product model
public function getAffiliateUrlAttribute()
{
    $affiliateId = config('services.themeforest.affiliate_id');
    return $this->themeforest_url . '?ref=' . $affiliateId;
}
```

### Store Expansion

The architecture supports full e-commerce functionality:

- **Shopping Cart**: Session-based or database cart
- **Order Management**: Complete order lifecycle management
- **User Accounts**: Purchase history, downloads, licenses
- **Email Notifications**: Order confirmations, receipts, download links
- **Digital Downloads**: Secure file delivery system
- **License Management**: Generate and validate product licenses

**Additional Tables Needed:**
- `carts` - Shopping cart items
- `orders` - Order records
- `order_items` - Order line items
- `downloads` - Download tracking
- `licenses` - Product licenses

## Troubleshooting

### Images Not Displaying

Ensure the storage link is created:

```bash
php artisan storage:link
```

### Tailwind Styles Not Loading

Rebuild assets:

```bash
npm run build
```

Or for development:

```bash
npm run dev
```

### Database Connection Error

Check your `.env` file database credentials match your MySQL setup.

### 404 Errors on Routes

Clear route cache:

```bash
php artisan route:clear
php artisan cache:clear
```

## Contributing

This is a complete project ready for deployment. Customize:

1. Update colors in `resources/css/app.css`
2. Modify layouts in `resources/views/layouts/`
3. Add new features following the existing structure
4. Extend admin panel as needed

## License

This project is open-source and available for use.

## Support

For issues or questions, refer to the Laravel documentation:
- [Laravel Docs](https://laravel.com/docs)
- [TailwindCSS Docs](https://tailwindcss.com/docs)

---

**Built with ❤️ using Laravel 11 + Blade + TailwindCSS**
