<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Post;
use App\Models\Product;
use App\Services\ContentParserService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ContentImportController extends Controller
{
    protected $parserService;

    public function __construct(ContentParserService $parserService)
    {
        $this->parserService = $parserService;
    }

    /**
     * Display the content import interface
     */
    public function index()
    {
        return view('admin.import.index');
    }

    /**
     * Parse the uploaded/pasted content
     */
    public function parse(Request $request)
    {
        $request->validate([
            'content' => 'required|string',
            'format' => 'nullable|string|in:auto,json,markdown,html,seeder,text',
        ]);

        $content = $request->input('content');
        $format = $request->input('format', 'auto');

        try {
            // Auto-detect format if needed
            if ($format === 'auto') {
                $format = $this->parserService->detectFormat($content);
            }

            // Parse the content
            $parsed = $this->parserService->parse($content, $format);

            // Store parsed data in session for preview
            session(['import_preview' => $parsed, 'import_format' => $format]);

            return response()->json([
                'success' => true,
                'data' => $parsed,
                'format' => $format,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to parse content: ' . $e->getMessage(),
            ], 422);
        }
    }

    /**
     * Display preview of parsed content
     */
    public function preview()
    {
        $parsed = session('import_preview');

        if (!$parsed) {
            return redirect()->route('admin.import.index')
                ->with('error', 'No content to preview. Please import content first.');
        }

        return view('admin.import.preview', [
            'parsed' => $parsed,
            'format' => session('import_format', 'auto'),
        ]);
    }

    /**
     * Store the imported content as Post or Product
     */
    public function store(Request $request)
    {
        $request->validate([
            'target_type' => 'required|string|in:post,product',
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'summary' => 'nullable|string',
            'content_html' => 'nullable|string',
            'content_markdown' => 'nullable|string',
            'affiliate_link' => 'nullable|url',
            'pros' => 'nullable|array',
            'cons' => 'nullable|array',
            'features' => 'nullable|array',
            'faqs' => 'nullable|array',
            'tables' => 'nullable|array',
            'cta' => 'nullable|string',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'keywords' => 'nullable|array',
            'category' => 'nullable|string|max:255',
            'price' => 'nullable|numeric|min:0',
            'status' => 'nullable|string|in:draft,published,active,inactive',
        ]);

        $targetType = $request->input('target_type');
        $parsed = session('import_preview');

        try {
            if ($targetType === 'post') {
                $post = $this->createPost($request, $parsed);
                session()->forget(['import_preview', 'import_format']);
                return redirect()->route('admin.content.posts.show', $post)
                    ->with('success', 'Content imported successfully as blog post!');
            } else {
                $product = $this->createProduct($request, $parsed);
                session()->forget(['import_preview', 'import_format']);
                return redirect()->route('admin.products.show', $product)
                    ->with('success', 'Content imported successfully as product!');
            }
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to save content: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Create a Post from imported content
     */
    protected function createPost(Request $request, $parsed)
    {
        $slug = $request->input('slug') ?: Str::slug($request->input('title'));

        // Check for unique slug
        $originalSlug = $slug;
        $counter = 1;
        while (Post::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Build content with pros/cons/features/faqs if available
        // Decode any HTML entities that might be in the content
        $content = $request->input('content_html') ?: $parsed['content_html'] ?? '';
        $content = html_entity_decode($content, ENT_QUOTES | ENT_HTML5, 'UTF-8');

        // Add pros/cons sections if available
        if (!empty($parsed['pros']) || !empty($request->input('pros'))) {
            $pros = $request->input('pros') ?: $parsed['pros'] ?? [];
            if (!empty($pros)) {
                $content .= '<h2>Pros</h2><ul>';
                foreach ($pros as $pro) {
                    // Convert array to string if needed, then escape
                    $proText = is_array($pro) ? (isset($pro['name']) ? $pro['name'] : (isset($pro['title']) ? $pro['title'] : json_encode($pro))) : (string)$pro;
                    $content .= '<li>' . htmlspecialchars($proText, ENT_QUOTES, 'UTF-8') . '</li>';
                }
                $content .= '</ul>';
            }
        }

        if (!empty($parsed['cons']) || !empty($request->input('cons'))) {
            $cons = $request->input('cons') ?: $parsed['cons'] ?? [];
            if (!empty($cons)) {
                $content .= '<h2>Cons</h2><ul>';
                foreach ($cons as $con) {
                    // Convert array to string if needed, then escape
                    $conText = is_array($con) ? (isset($con['name']) ? $con['name'] : (isset($con['title']) ? $con['title'] : json_encode($con))) : (string)$con;
                    $content .= '<li>' . htmlspecialchars($conText, ENT_QUOTES, 'UTF-8') . '</li>';
                }
                $content .= '</ul>';
            }
        }

        // Add features section if available
        if (!empty($parsed['features']) || !empty($request->input('features'))) {
            $features = $request->input('features') ?: $parsed['features'] ?? [];
            if (!empty($features)) {
                $content .= '<h2>Features</h2><ul>';
                foreach ($features as $feature) {
                    // Handle feature objects with title and description
                    if (is_array($feature)) {
                        $featureTitle = $feature['title'] ?? $feature['name'] ?? '';
                        $featureDesc = $feature['description'] ?? $feature['desc'] ?? '';
                        if ($featureTitle && $featureDesc) {
                            $content .= '<li><strong>' . htmlspecialchars($featureTitle, ENT_QUOTES, 'UTF-8') . '</strong>: ' . htmlspecialchars($featureDesc, ENT_QUOTES, 'UTF-8') . '</li>';
                        } elseif ($featureTitle) {
                            $content .= '<li>' . htmlspecialchars($featureTitle, ENT_QUOTES, 'UTF-8') . '</li>';
                        } else {
                            $content .= '<li>' . htmlspecialchars(json_encode($feature), ENT_QUOTES, 'UTF-8') . '</li>';
                        }
                    } else {
                        $content .= '<li>' . htmlspecialchars((string)$feature, ENT_QUOTES, 'UTF-8') . '</li>';
                    }
                }
                $content .= '</ul>';
            }
        }

        // Add FAQs section if available
        if (!empty($parsed['faqs']) || !empty($request->input('faqs'))) {
            $faqs = $request->input('faqs') ?: $parsed['faqs'] ?? [];
            if (!empty($faqs)) {
                $content .= '<h2>Frequently Asked Questions</h2>';
                foreach ($faqs as $faq) {
                    $question = is_array($faq) ? ($faq['question'] ?? $faq['q'] ?? '') : '';
                    $answer = is_array($faq) ? ($faq['answer'] ?? $faq['a'] ?? '') : $faq;
                    if ($question && $answer) {
                        $content .= '<h3>' . htmlspecialchars($question, ENT_QUOTES, 'UTF-8') . '</h3>';
                        $content .= '<p>' . htmlspecialchars($answer, ENT_QUOTES, 'UTF-8') . '</p>';
                    }
                }
            }
        }

        // Extract affiliate link and CTA text (check both request and parsed data)
        $affiliateLink = $request->input('affiliate_link') ?: $parsed['affiliate_link'] ?? '';
        $ctaText = $request->input('cta_section') ?: $parsed['cta_section'] ?? $parsed['cta'] ?? '';
        
        // Add CTA section to content if available (with affiliate link if provided)
        if (!empty($ctaText) || !empty($affiliateLink)) {
            $content .= '<div class="cta-section" style="margin-top: 3rem; padding: 2rem; background: rgba(0, 180, 216, 0.1); border: 2px solid #00B4D8; border-radius: 0.5rem; text-align: center;">';
            if (!empty($ctaText)) {
                $content .= '<h2 style="color: #00B4D8; margin-bottom: 1rem;">' . htmlspecialchars($ctaText, ENT_QUOTES, 'UTF-8') . '</h2>';
            }
            if (!empty($affiliateLink)) {
                $content .= '<a href="' . htmlspecialchars($affiliateLink, ENT_QUOTES, 'UTF-8') . '" target="_blank" rel="nofollow" style="display: inline-block; padding: 1rem 2rem; background: #00B4D8; color: white; text-decoration: none; border-radius: 0.5rem; font-weight: bold; margin-top: 1rem;">Get It Now</a>';
            }
            $content .= '</div>';
        }

        // Create post with all fields including affiliate_link and cta_section
        $post = Post::create([
            'title' => $request->input('title'),
            'slug' => $slug,
            'excerpt' => $request->input('summary') ?: $parsed['summary'] ?? '',
            'content' => $content,
            'category' => $request->input('category') ?: null,
            'affiliate_link' => $affiliateLink, // Save to database
            'cta_section' => $ctaText, // Save to database
            'meta_title' => $request->input('meta_title') ?: $parsed['seo']['meta_title'] ?? $request->input('title'),
            'meta_description' => $request->input('meta_description') ?: $parsed['seo']['meta_description'] ?? '',
            'published_at' => $request->input('status') === 'published' ? now() : null,
        ]);

        return $post;
    }

    /**
     * Create a Product from imported content
     */
    protected function createProduct(Request $request, $parsed)
    {
        $slug = $request->input('slug') ?: Str::slug($request->input('title'));

        // Check for unique slug
        $originalSlug = $slug;
        $counter = 1;
        while (Product::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Build description with features
        $description = $request->input('content_html') ?: $parsed['content_html'] ?? '';
        
        if (!empty($parsed['features']) || !empty($request->input('features'))) {
            $features = $request->input('features') ?: $parsed['features'] ?? [];
            if (!empty($features)) {
                $description .= '<h3>Key Features</h3><ul>';
                foreach ($features as $feature) {
                    $description .= '<li>' . htmlspecialchars($feature) . '</li>';
                }
                $description .= '</ul>';
            }
        }

        $product = Product::create([
            'name' => $request->input('title'),
            'slug' => $slug,
            'description' => $description,
            'short_description' => $request->input('summary') ?: $parsed['summary'] ?? '',
            'price' => $request->input('price') ?: 0,
            'status' => $request->input('status') ?: 'active',
            'featured' => $request->input('featured', false),
            'themeforest_url' => $request->input('affiliate_link') ?: $parsed['affiliate_link'] ?? null,
        ]);

        return $product;
    }
}

