<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Post;
use App\Models\Product;
use App\Models\Contact;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Display the admin dashboard with analytics and statistics.
     */
    public function index()
    {
        // Content Statistics
        $stats = [
            'posts' => [
                'total' => Post::count(),
                'published' => Post::published()->count(),
                'drafts' => Post::whereNull('published_at')->count(),
                'this_month' => Post::whereMonth('created_at', now()->month)->count(),
            ],
            'products' => [
                'total' => Product::count(),
                'active' => Product::active()->count(),
                'featured' => Product::where('status', 'active')->count(), // Add featured field later
                'this_month' => Product::whereMonth('created_at', now()->month)->count(),
            ],
            'contacts' => [
                'total' => Contact::count(),
                'unread' => Contact::whereNull('read_at')->count(),
                'this_month' => Contact::whereMonth('created_at', now()->month)->count(),
            ],
            'users' => [
                'total' => User::count(),
                'this_month' => User::whereMonth('created_at', now()->month)->count(),
            ],
        ];
        
        // Recent Activity
        $recentPosts = Post::latest()->limit(5)->get();
        $recentProducts = Product::latest()->limit(5)->get();
        $recentContacts = Contact::latest()->limit(5)->get();
        
        // Chart Data (Last 7 days)
        $chartData = $this->getChartData();
        
        return view('admin.dashboard.index', compact('stats', 'recentPosts', 'recentProducts', 'recentContacts', 'chartData'));
    }
    
    /**
     * Get chart data for the last 7 days.
     */
    private function getChartData()
    {
        $days = collect(range(6, 0))->map(function ($day) {
            return Carbon::now()->subDays($day)->format('Y-m-d');
        });
        
        $postsData = Post::select(DB::raw('DATE(created_at) as date'), DB::raw('count(*) as count'))
            ->where('created_at', '>=', Carbon::now()->subDays(7))
            ->groupBy('date')
            ->pluck('count', 'date');
        
        $productsData = Product::select(DB::raw('DATE(created_at) as date'), DB::raw('count(*) as count'))
            ->where('created_at', '>=', Carbon::now()->subDays(7))
            ->groupBy('date')
            ->pluck('count', 'date');
        
        return [
            'labels' => $days->map(fn($date) => Carbon::parse($date)->format('M d'))->toArray(),
            'posts' => $days->map(fn($date) => $postsData[$date] ?? 0)->toArray(),
            'products' => $days->map(fn($date) => $productsData[$date] ?? 0)->toArray(),
        ];
    }
}
