<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Media;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class MediaController extends Controller
{
    public function index()
    {
        $media = Media::latest()->paginate(24);
        return view('admin.media.index', compact('media'));
    }

    public function upload(Request $request)
    {
        $request->validate([
            'files.*' => 'required|file|max:10240', // 10MB max
        ]);

        $uploaded = [];
        
        foreach ($request->file('files') as $file) {
            $path = $file->store('media', 'public');
            $type = $this->determineType($file->getMimeType());
            
            $media = Media::create([
                'name' => $file->getClientOriginalName(),
                'file_name' => $file->hashName(),
                'mime_type' => $file->getMimeType(),
                'path' => $path,
                'disk' => 'public',
                'size' => $file->getSize(),
                'type' => $type,
                'user_id' => auth()->id(),
            ]);
            
            $uploaded[] = $media;
        }

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'media' => $uploaded
            ]);
        }

        return redirect()->back()->with('success', count($uploaded) . ' file(s) uploaded successfully');
    }

    public function destroy(Media $media)
    {
        Storage::disk($media->disk)->delete($media->path);
        $media->delete();
        
        return redirect()->back()->with('success', 'Media deleted successfully');
    }

    public function browser()
    {
        $media = Media::where('type', 'image')->latest()->get();
        return view('admin.media.browser', compact('media'));
    }

    private function determineType($mimeType)
    {
        if (str_starts_with($mimeType, 'image/')) {
            return 'image';
        } elseif (str_starts_with($mimeType, 'video/')) {
            return 'video';
        } else {
            return 'document';
        }
    }
}
