<?php

namespace App\Services\Parsers;

use Illuminate\Support\Str;

class MarkdownParser implements ParserInterface
{
    public function parse(string $content): array
    {
        $lines = explode("\n", $content);
        $title = '';
        $summary = '';
        $contentHtml = '';
        $pros = [];
        $cons = [];
        $features = [];
        $faqs = [];
        $affiliateLink = '';
        $cta = '';

        $inPros = false;
        $inCons = false;
        $inFeatures = false;
        $inFaqs = false;

        foreach ($lines as $line) {
            $line = trim($line);

            // Extract title (first H1)
            if (empty($title) && preg_match('/^#\s+(.+)$/', $line, $matches)) {
                $title = $matches[1];
                continue;
            }

            // Extract summary (first paragraph or after title)
            if (empty($summary) && !empty($line) && !preg_match('/^[#*\-\[\]]/', $line)) {
                $summary = $line;
                continue;
            }

            // Detect sections
            if (preg_match('/^##?\s+(pros|advantages|benefits)/i', $line)) {
                $inPros = true;
                $inCons = false;
                $inFeatures = false;
                $inFaqs = false;
                continue;
            }

            if (preg_match('/^##?\s+(cons|disadvantages|drawbacks)/i', $line)) {
                $inCons = true;
                $inPros = false;
                $inFeatures = false;
                $inFaqs = false;
                continue;
            }

            if (preg_match('/^##?\s+(features|key features)/i', $line)) {
                $inFeatures = true;
                $inPros = false;
                $inCons = false;
                $inFaqs = false;
                continue;
            }

            if (preg_match('/^##?\s+(faq|frequently asked questions|questions)/i', $line)) {
                $inFaqs = true;
                $inPros = false;
                $inCons = false;
                $inFeatures = false;
                continue;
            }

            // Extract list items
            if (preg_match('/^[-*+]\s+(.+)$/', $line, $matches)) {
                $item = $matches[1];
                if ($inPros) {
                    $pros[] = $item;
                } elseif ($inCons) {
                    $cons[] = $item;
                } elseif ($inFeatures) {
                    $features[] = $item;
                }
                continue;
            }

            // Extract FAQ items
            if ($inFaqs && preg_match('/^###?\s+(.+)$/', $line, $matches)) {
                $faqs[] = ['question' => $matches[1], 'answer' => ''];
                continue;
            }

            if ($inFaqs && !empty($line) && !empty($faqs)) {
                $lastIndex = count($faqs) - 1;
                if (empty($faqs[$lastIndex]['answer'])) {
                    $faqs[$lastIndex]['answer'] = $line;
                } else {
                    $faqs[$lastIndex]['answer'] .= ' ' . $line;
                }
                continue;
            }

            // Extract affiliate link
            if (preg_match('/\[([^\]]+)\]\(([^)]+)\)/', $line, $matches)) {
                if (stripos($matches[1], 'buy') !== false || stripos($matches[1], 'affiliate') !== false) {
                    $affiliateLink = $matches[2];
                }
            }

            // Extract CTA
            if (preg_match('/cta[:\s]+(.+)/i', $line, $matches)) {
                $cta = $matches[1];
            }

            // Add to content
            if (!empty($line) && !$inPros && !$inCons && !$inFeatures && !$inFaqs) {
                $contentHtml .= $this->markdownToHtml($line) . "\n";
            }
        }

        return [
            'title' => $title,
            'slug' => $title ? Str::slug($title) : '',
            'summary' => $summary,
            'content_html' => $contentHtml,
            'content_markdown' => $content,
            'affiliate_link' => $affiliateLink,
            'pros' => $pros,
            'cons' => $cons,
            'features' => $features,
            'faqs' => $faqs,
            'tables' => [],
            'cta' => $cta,
            'seo' => [
                'meta_title' => $title,
                'meta_description' => $summary,
                'keywords' => [],
            ],
        ];
    }

    protected function markdownToHtml(string $markdown): string
    {
        // Convert markdown to HTML (basic conversion)
        $html = $markdown;

        // Headers
        $html = preg_replace('/^### (.+)$/m', '<h3>$1</h3>', $html);
        $html = preg_replace('/^## (.+)$/m', '<h2>$1</h2>', $html);
        $html = preg_replace('/^# (.+)$/m', '<h1>$1</h1>', $html);

        // Bold
        $html = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $html);
        $html = preg_replace('/__(.+?)__/', '<strong>$1</strong>', $html);

        // Italic
        $html = preg_replace('/\*(.+?)\*/', '<em>$1</em>', $html);
        $html = preg_replace('/_(.+?)_/', '<em>$1</em>', $html);

        // Links
        $html = preg_replace('/\[([^\]]+)\]\(([^)]+)\)/', '<a href="$2">$1</a>', $html);

        // Line breaks
        $html = nl2br($html);

        return $html;
    }
}

