<?php

namespace App\Services\Parsers;

use Illuminate\Support\Str;

class TextParser implements ParserInterface
{
    public function parse(string $content): array
    {
        $lines = explode("\n", $content);
        $title = '';
        $summary = '';
        $contentHtml = '';
        $pros = [];
        $cons = [];
        $features = [];
        $faqs = [];
        $affiliateLink = '';
        $cta = '';

        $inPros = false;
        $inCons = false;
        $inFeatures = false;
        $inFaqs = false;

        foreach ($lines as $line) {
            $line = trim($line);

            if (empty($line)) {
                continue;
            }

            // Extract title (first non-empty line or line with "Title:")
            if (empty($title)) {
                if (preg_match('/^title[:\s]+(.+)$/i', $line, $matches)) {
                    $title = trim($matches[1]);
                    continue;
                } elseif (empty($title) && strlen($line) < 100) {
                    $title = $line;
                    continue;
                }
            }

            // Extract summary
            if (empty($summary) && preg_match('/^(?:summary|excerpt|description)[:\s]+(.+)$/i', $line, $matches)) {
                $summary = trim($matches[1]);
                continue;
            }

            // Detect sections
            if (preg_match('/^(?:pros|advantages|benefits)[:\s]*$/i', $line)) {
                $inPros = true;
                $inCons = false;
                $inFeatures = false;
                $inFaqs = false;
                continue;
            }

            if (preg_match('/^(?:cons|disadvantages|drawbacks)[:\s]*$/i', $line)) {
                $inCons = true;
                $inPros = false;
                $inFeatures = false;
                $inFaqs = false;
                continue;
            }

            if (preg_match('/^(?:features|key features)[:\s]*$/i', $line)) {
                $inFeatures = true;
                $inPros = false;
                $inCons = false;
                $inFaqs = false;
                continue;
            }

            if (preg_match('/^(?:faq|frequently asked questions|questions)[:\s]*$/i', $line)) {
                $inFaqs = true;
                $inPros = false;
                $inCons = false;
                $inFeatures = false;
                continue;
            }

            // Extract list items (lines starting with dash, number, or bullet)
            if (preg_match('/^[-*•]\s+(.+)$/', $line, $matches) || preg_match('/^\d+[.)]\s+(.+)$/', $line, $matches)) {
                $item = trim($matches[1]);
                if ($inPros) {
                    $pros[] = $item;
                } elseif ($inCons) {
                    $cons[] = $item;
                } elseif ($inFeatures) {
                    $features[] = $item;
                }
                continue;
            }

            // Extract FAQ items (Q: and A: pattern)
            if (preg_match('/^q[:\s]+(.+)$/i', $line, $matches)) {
                $faqs[] = ['question' => trim($matches[1]), 'answer' => ''];
                continue;
            }

            if (preg_match('/^a[:\s]+(.+)$/i', $line, $matches) && !empty($faqs)) {
                $lastIndex = count($faqs) - 1;
                $faqs[$lastIndex]['answer'] = trim($matches[1]);
                continue;
            }

            // Extract affiliate link
            if (preg_match('/(?:affiliate|buy)[:\s]+(https?:\/\/[^\s]+)/i', $line, $matches)) {
                $affiliateLink = trim($matches[1]);
            }

            // Extract CTA
            if (preg_match('/cta[:\s]+(.+)/i', $line, $matches)) {
                $cta = trim($matches[1]);
            }

            // Add to content
            if (!empty($line) && !$inPros && !$inCons && !$inFeatures && !$inFaqs) {
                $contentHtml .= '<p>' . htmlspecialchars($line) . '</p>' . "\n";
            }
        }

        // If no title found, use first line
        if (empty($title) && !empty($lines)) {
            $title = trim($lines[0]);
        }

        // If no summary found, use second line or first paragraph
        if (empty($summary) && count($lines) > 1) {
            $summary = trim($lines[1]);
        }

        return [
            'title' => $title,
            'slug' => $title ? Str::slug($title) : '',
            'summary' => $summary,
            'content_html' => $contentHtml,
            'content_markdown' => $content,
            'affiliate_link' => $affiliateLink,
            'pros' => $pros,
            'cons' => $cons,
            'features' => $features,
            'faqs' => $faqs,
            'tables' => [],
            'cta' => $cta,
            'seo' => [
                'meta_title' => $title,
                'meta_description' => $summary,
                'keywords' => [],
            ],
        ];
    }
}

